<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Admin;

use Erlage\Photogram\Tools\Crypto;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Exceptions\SessionException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Admin\AdminTable;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\Admin\AdminFinder;

final class AdminAuth extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | session request [auto]
    |--------------------------------------------------------------------------
    */

    public static function session(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | ensure admin is authenticated
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | return authed admin
            |--------------------------------------------------------------------------
            */

            self::addToResponse(AdminTable::getTableName(), self::$authedAdminModel -> getPrivateMap());
        });
    }

    /*
    |--------------------------------------------------------------------------
    | login request {username, password}
    |--------------------------------------------------------------------------
    */

    public static function login(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $usernameFromReq = self::$request -> findKey(
                AdminTable::USERNAME,
                RequestTable::PAYLOAD,
                AdminTable::TABLE_NAME
            );

            $passwordFromReq = self::$request -> findKey(
                AdminTable::PASSWORD,
                RequestTable::PAYLOAD,
                AdminTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::D_ERROR_ADMIN_NOT_MATCHED_MSG, $usernameFromReq, $passwordFromReq);

            /*
            |--------------------------------------------------------------------------
            | try fetching the admin using the provided username
            |--------------------------------------------------------------------------
            */

            $finder = (new AdminFinder())
                -> setUsername($usernameFromReq)
                -> find();

            if ($finder -> notFound())
            {
                throw new RequestException(ResponseConstants::D_ERROR_ADMIN_NOT_MATCHED_MSG);
            }

            $adminModel = $finder -> popModelFromResults();

            /*
            |--------------------------------------------------------------------------
            | verify the provided password
            |--------------------------------------------------------------------------
            */

            if ( ! Crypto::verifyClearPasswordWithDatabaseHash($passwordFromReq, $adminModel -> getPassword()))
            {
                throw new RequestException(ResponseConstants::D_ERROR_ADMIN_NOT_MATCHED_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | do server side login
            |--------------------------------------------------------------------------
            */

            self::$adminSession -> loginAs($adminModel);

            self::adminAuthenticate();

            /*
            |--------------------------------------------------------------------------
            | ensuring call make sure that user is allowed to construct sessions
            | this will throw in case of ban, email not verified etc
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | add admin to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(AdminTable::getTableName(), $adminModel -> getPrivateMap());
        });
    }

    public static function logout(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            self::$adminSession -> logout();

            throw new SessionException(ResponseConstants::SUCCESS_MSG);
        });
    }

    public static function update(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $passwordFromReq = self::$request -> findKey(
                AdminTable::PASSWORD,
                RequestTable::PAYLOAD,
                AdminTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::D_ERROR_ADMIN_NOT_MATCHED_MSG, $passwordFromReq);

            /*
            |--------------------------------------------------------------------------
            | verify the provided password
            |--------------------------------------------------------------------------
            */

            if ( ! Crypto::verifyClearPasswordWithDatabaseHash($passwordFromReq, self::$authedAdminModel -> getPassword()))
            {
                throw new RequestException(ResponseConstants::D_ERROR_ADMIN_NOT_MATCHED_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | get data to update
            |--------------------------------------------------------------------------
            */

            $newUsernameFromReq = self::$request -> findKey(
                AdminTable::USERNAME,
                RequestTable::PAYLOAD,
                AdminTable::TABLE_NAME
            );

            $newPasswordFromReq = self::$request -> findKey(
                AdminTable::EXTRA_NEW_PASSWORD,
                RequestTable::PAYLOAD,
                AdminTable::TABLE_NAME
            );

            $updateSet = array();

            /*
            |--------------------------------------------------------------------------
            | if change username
            |--------------------------------------------------------------------------
            */

            if (self::isAvailable($newUsernameFromReq) && \strlen($newUsernameFromReq) > 0)
            {
                $updateSet[AdminTable::USERNAME] = $newUsernameFromReq;
            }

            /*
            |--------------------------------------------------------------------------
            | if change password
            |--------------------------------------------------------------------------
            */

            if (self::isAvailable($newPasswordFromReq) && \strlen($newPasswordFromReq) > 0)
            {
                $updateSet[AdminTable::PASSWORD] = $newPasswordFromReq;
                $updateSet[AdminTable::META_ACCESS] = '';
            }

            /*
            |--------------------------------------------------------------------------
            | do update
            |--------------------------------------------------------------------------
            */

            self::$authedAdminModel -> update($updateSet);

            self::$authedAdminModel -> save();

            /*
            |--------------------------------------------------------------------------
            | add admin to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(AdminTable::getTableName(), self::$authedAdminModel -> getDataMap());
        });
    }
}
